<?php

namespace ArchiElite\EventsBooking\Http\Controllers\Accounts;

use ArchiElite\EventsBooking\Facades\InvoiceHelper;
use ArchiElite\EventsBooking\Forms\Fronts\Accounts\AccountForm;
use ArchiElite\EventsBooking\Forms\Fronts\Auth\ChangePasswordForm;
use ArchiElite\EventsBooking\Http\Requests\Fronts\ChangePasswordRequest;
use ArchiElite\EventsBooking\Http\Requests\Fronts\EditAccountRequest;
use ArchiElite\EventsBooking\Models\Currency;
use ArchiElite\EventsBooking\Models\Invoice;
use Botble\Base\Facades\BaseHelper;
use Botble\Base\Http\Controllers\BaseController;
use Botble\SeoHelper\Facades\SeoHelper;
use Botble\Theme\Facades\Theme;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Str;

class PublicController extends BaseController
{
    public function __construct()
    {
        Theme::asset()
            ->add('account-style', 'vendor/core/plugins/events-booking/css/account.css', ['bootstrap-css']);
    }

    public function getOverview()
    {
        SeoHelper::setTitle(__('Account information'));

        Theme::breadcrumb()
            ->add(__('Account information'), route('account.overview'));

        $account = auth('account')->user();

        return Theme::scope(
            'events-booking.accounts.overview',
            compact('account'),
            'plugins/events-booking::themes.accounts.overview'
        )
            ->render();
    }

    public function getEditAccount()
    {
        SeoHelper::setTitle(__('Profile'));

        Theme::asset()
            ->add(
                'datepicker-style',
                'vendor/core/core/base/libraries/bootstrap-datepicker/css/bootstrap-datepicker3.min.css',
                ['bootstrap']
            );
        Theme::asset()
            ->container('footer')
            ->add(
                'datepicker-js',
                'vendor/core/core/base/libraries/bootstrap-datepicker/js/bootstrap-datepicker.min.js',
                ['jquery']
            );

        Theme::breadcrumb()
            ->add(__('Profile'), route('account.edit-account'));

        $account = auth('account')->user();

        $form = AccountForm::createFromModel($account);

        return Theme::scope(
            'events-booking.accounts.edit-account',
            compact('form'),
            'plugins/events-booking::themes.accounts.edit-account'
        )
            ->render();
    }

    public function postEditAccount(EditAccountRequest $request)
    {
        $account = auth('account')->user();

        AccountForm::createFromModel($account)
            ->setRequest($request)
            ->saving(function (AccountForm $form) {
                $model = $form->getModel();
                $request = $form->getRequest();

                $model->fill($request->except(['email']));

                $model->dob = Carbon::createFromFormat(BaseHelper::getDateFormat(), $request->input('dob'));

                $model->save();
            });

        return $this
            ->httpResponse()
            ->setNextUrl(route('account.edit-account'))
            ->setMessage(__('Update profile successfully!'));
    }

    public function getChangePassword()
    {
        SeoHelper::setTitle(__('Change Password'));

        Theme::breadcrumb()
            ->add(__('Change Password'), route('account.change-password'));

        $form = ChangePasswordForm::create();

        return Theme::scope(
            'events-booking.accounts.change-password',
            compact('form'),
            'plugins/events-booking::themes.accounts.change-password'
        )->render();
    }

    public function postChangePassword(ChangePasswordRequest $request)
    {
        $user = Auth::guard('account')->user();

        ChangePasswordForm::createFromModel($user)
            ->setRequest($request)
            ->saving(function (ChangePasswordForm $form) {
                $model = $form->getModel();
                $request = $form->getRequest();

                $model->update([
                    'password' => Hash::make($request->input('password')),
                ]);
            });

        return $this
            ->httpResponse()
            ->setMessage(trans('core/acl::users.password_update_success'));
    }

    public function getGenerateInvoice(int|string $invoiceId, Request $request)
    {
        $invoice = Invoice::query()->findOrFail($invoiceId);

        abort_unless($this->canViewInvoice($invoice), 404);

        if ($request->input('type') === 'print') {
            return InvoiceHelper::streamInvoice($invoice);
        }

        return InvoiceHelper::downloadInvoice($invoice);
    }

    protected function canViewInvoice(Invoice $invoice): bool
    {
        return auth('account')->id() == $invoice->payment->customer_id;
    }

    public function switchCurrency(string $title, Request $request)
    {
        if (empty($title)) {
            $title = $request->input('currency');
        }

        if (! $title) {
            return $this->httpResponse();
        }

        $currency = Currency::query()->where('title', $title)->first();

        if ($currency) {
            cms_currency()->setApplicationCurrency($currency);
        }

        $url = URL::previous();

        if (! $url || $url === URL::current()) {
            return $this
                ->httpResponse()
                ->setNextUrl(BaseHelper::getHomepageUrl());
        }

        if (Str::contains($url, ['min_price', 'max_price'])) {
            $url = preg_replace('/&min_price=[0-9]+/', '', $url);
            $url = preg_replace('/&max_price=[0-9]+/', '', $url);
        }

        return $this
            ->httpResponse()
            ->setNextUrl($url);
    }
}
