<?php

namespace ArchiElite\EventsBooking\Http\Controllers;

use ArchiElite\EventsBooking\Http\Requests\CouponRequest;
use ArchiElite\EventsBooking\Models\Coupon;
use ArchiElite\EventsBooking\Tables\CouponTable;
use Botble\Base\Events\CreatedContentEvent;
use Botble\Base\Events\UpdatedContentEvent;
use Botble\Base\Facades\Assets;
use Botble\Base\Http\Actions\DeleteResourceAction;
use Botble\Base\Http\Controllers\BaseController;
use Botble\JsValidation\Facades\JsValidator;
use Illuminate\Support\Str;

class CouponController extends BaseController
{
    public function __construct()
    {
        $this
            ->breadcrumb()
            ->add(trans('plugins/events-booking::coupon.name'), route('events-booking.coupons.index'));
    }

    public function index(CouponTable $discountTable)
    {
        $this->pageTitle(trans('plugins/events-booking::coupon.name'));

        return $discountTable->renderTable();
    }

    public function create()
    {
        $this->pageTitle(trans('plugins/events-booking::coupon.form.create'));

        Assets::usingVueJS()
            ->addStyles('timepicker')
            ->addScripts(['timepicker', 'form-validation'])
            ->addScriptsDirectly('vendor/core/plugins/events-booking/js/coupon.js');

        $jsValidator = JsValidator::formRequest(CouponRequest::class);

        $coupon = new Coupon();

        return view('plugins/events-booking::coupons.create', compact('jsValidator', 'coupon'));
    }

    public function store(CouponRequest $request)
    {
        $coupon = Coupon::query()->create(array_merge($request->validated(), $request->has('never_expired') ? [] : [
            'expires_date' => $request->date('expires_date')->setTimeFrom($request->input('expires_time')),
        ]));

        event(new CreatedContentEvent(COUPON_MODULE_SCREEN_NAME, $request, $coupon));

        return $this
            ->httpResponse()
            ->setMessage(trans('plugins/events-booking::coupon.created_message'))
            ->setNextUrl(route('events-booking.coupons.edit', $coupon));
    }

    public function edit(Coupon $coupon)
    {
        $this->pageTitle(trans('plugins/events-booking::coupon.form.edit', ['name' => $coupon->code]));

        Assets::usingVueJS()
            ->addStyles('timepicker')
            ->addScripts(['timepicker', 'form-validation'])
            ->addScriptsDirectly('vendor/core/plugins/events-booking/js/coupon.js');

        $jsValidator = JsValidator::formRequest(CouponRequest::class);

        return view('plugins/events-booking::coupons.edit', compact('coupon', 'jsValidator'));
    }

    public function update(Coupon $coupon, CouponRequest $request)
    {
        $coupon->update(
            array_merge(
                $request->validated(),
                $request->has('never_expired')
                    ? ['expires_date' => null]
                    : ['expires_date' => $request->date('expires_date')->setTimeFrom($request->input('expires_time'))]
            )
        );

        event(new UpdatedContentEvent(COUPON_MODULE_SCREEN_NAME, $request, $coupon));

        return $this
            ->httpResponse()
            ->withUpdatedSuccessMessage();
    }

    public function destroy(Coupon $coupon)
    {
        return DeleteResourceAction::make($coupon);
    }

    public function generateCouponCode()
    {
        do {
            $code = strtoupper(Str::random(12));
        } while (Coupon::query()->where('code', $code)->exists());

        return $this
            ->httpResponse()
            ->setData($code);
    }
}
