<?php

namespace ArchiElite\EventsBooking\Models;

use ArchiElite\EventsBooking\Enums\AccountStatusEnum;
use ArchiElite\EventsBooking\Enums\AccountTypeEnum;
use ArchiElite\EventsBooking\Notifications\ConfirmEmailNotification;
use ArchiElite\EventsBooking\Notifications\ResetPasswordNotification;
use Botble\Base\Models\BaseModel;
use Illuminate\Auth\Authenticatable;
use Illuminate\Auth\MustVerifyEmail;
use Illuminate\Auth\Passwords\CanResetPassword;
use Illuminate\Contracts\Auth\Access\Authorizable as AuthorizableContract;
use Illuminate\Contracts\Auth\Authenticatable as AuthenticatableContract;
use Illuminate\Contracts\Auth\CanResetPassword as CanResetPasswordContract;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Foundation\Auth\Access\Authorizable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class Account extends BaseModel implements AuthenticatableContract, AuthorizableContract, CanResetPasswordContract
{
    use Authenticatable;
    use Authorizable;
    use CanResetPassword;
    use HasApiTokens;
    use MustVerifyEmail;
    use Notifiable;

    protected $table = 'eb_accounts';

    protected $fillable = [
        'name',
        'email',
        'avatar',
        'phone',
        'dob',
        'type',
        'status',
        'remember_token',
        'password',
    ];

    protected $hidden = [
        'dob' => 'date',
        'password',
        'remember_token',
    ];

    protected $casts = [
        'dob' => 'date',
        'password' => 'hashed',
        'status' => AccountStatusEnum::class,
        'type' => AccountTypeEnum::class,
    ];

    public function sendEmailVerificationNotification(): void
    {
        $this->notify(new ConfirmEmailNotification());
    }

    public function sendPasswordResetNotification($token): void
    {
        $this->notify(new ResetPasswordNotification($token));
    }

    public function getAvatarUrlAttribute()
    {
        return $this->avatar ?: 'https://ui-avatars.com/api/?name=' . urlencode($this->name) . '&color=7F9CF5&background=EBF4FF';
    }

    public function events(): BelongsToMany
    {
        return $this->belongsToMany(Event::class, 'eb_event_attendee', 'attendee_id', 'event_id');
    }
}
