<?php

namespace ArchiElite\EventsBooking\Models;

use ArchiElite\EventsBooking\Enums\EventStatusEnum;
use Botble\Base\Models\BaseModel;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Event extends BaseModel
{
    protected $table = 'eb_events';

    protected $fillable = [
        'name',
        'unique_id',
        'tax_id',
        'price',
        'account_id',
        'description',
        'short_description',
        'content',
        'image',
        'address',
        'country_id',
        'state_id',
        'city_id',
        'latitude',
        'longitude',
        'started_at',
        'ended_at',
        'seats',
        'views',
        'status',
        'is_featured',
        'booking_closed_at',
    ];

    protected $casts = [
        'started_at' => 'datetime',
        'ended_at' => 'datetime',
        'booking_closed_at' => 'datetime',
        'seats' => 'int',
        'views' => 'int',
        'is_featured' => 'bool',
        'status' => EventStatusEnum::class,
    ];

    protected $appends = ['booked_seats'];

    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class, 'currency_id');
    }

    public function organizer(): BelongsTo
    {
        return $this->belongsTo(Account::class, 'account_id');
    }

    public function speakers(): BelongsToMany
    {
        return $this->belongsToMany(Account::class, 'eb_event_speaker', 'event_id', 'speaker_id')
            ->withPivot('priority', 'title', 'subtitle', 'started_at', 'ended_at')
            ->orderByPivot('priority');
    }

    public function attendees(): BelongsToMany
    {
        return $this->belongsToMany(Account::class, 'eb_event_attendee', 'event_id', 'attendee_id')->withPivot('checked_in_at', 'hash');
    }

    public function getEventTimeAttribute(): string
    {
        $startDate = $this->started_at->format('d F Y');
        $startTime = $this->started_at->format('g:ia');

        if (! $this->ended_at) {
            return "{$startDate} - {$startTime} to ∞";
        }

        $endDate = $this->ended_at->format('d F Y');
        $endTime = $this->ended_at->format('g:ia');

        if ($startDate === $endDate) {
            return "{$startDate} - {$startTime} to {$endTime}";
        } else {
            return "{$startDate} - {$startTime} to {$endDate} - {$endTime}";
        }
    }

    public function getBookedSeatsAttribute(): int
    {
        return $this->attendees()->count();
    }

    public function tax(): BelongsTo
    {
        return $this->belongsTo(Tax::class, 'tax_id');
    }

    public function orders(): HasMany
    {
        return $this->hasMany(Order::class, 'event_id');
    }
}
