<?php

namespace ArchiElite\EventsBooking\Providers;

use ArchiElite\EventsBooking\Enums\OrderStatusEnum;
use ArchiElite\EventsBooking\Events\OrderCompleted;
use ArchiElite\EventsBooking\Models\Account;
use ArchiElite\EventsBooking\Models\Event;
use ArchiElite\EventsBooking\Models\Order;
use ArchiElite\EventsBooking\Services\OrderService;
use ArchiElite\EventsBooking\Supports\TwigExtension;
use Botble\Base\Facades\Html;
use Botble\Base\Supports\ServiceProvider;
use Botble\Base\Supports\TwigCompiler;
use Botble\Page\Models\Page;
use Botble\Page\Tables\PageTable;
use Botble\Payment\Enums\PaymentMethodEnum;
use Botble\Payment\Enums\PaymentStatusEnum;
use Botble\Payment\Supports\PaymentHelper;
use Botble\Theme\Events\RenderingThemeOptionSettings;
use Botble\Theme\Facades\Theme;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;

class HookServiceProvider extends ServiceProvider
{
    public function boot(): void
    {
        if (defined('PAYMENT_FILTER_REDIRECT_URL')) {
            add_filter(PAYMENT_FILTER_REDIRECT_URL, function ($checkoutToken) {
                return route('public.booking.information', $checkoutToken ?: session('checkout_token'));
            }, 123);
        }

        if (defined('PAYMENT_FILTER_CANCEL_URL')) {
            add_filter(PAYMENT_FILTER_CANCEL_URL, function ($checkoutToken) {
                return route('public.booking.form', ['token' => $checkoutToken ?: session('checkout_token')] + ['error' => true, 'error_type' => 'payment']);
            }, 123);
        }

        if (defined('PAYMENT_ACTION_PAYMENT_PROCESSED')) {
            add_action(PAYMENT_ACTION_PAYMENT_PROCESSED, function ($data) {
                $orderIds = $data['order_id'];
                $orderId = Arr::first($orderIds);

                PaymentHelper::storeLocalPayment($data);

                return $this->app->make(OrderService::class)->processBooking($orderId, $data['charge_id']);
            });
        }

        if (defined('PAYMENT_FILTER_PAYMENT_DATA')) {
            add_filter(PAYMENT_FILTER_PAYMENT_DATA, function (array $data, Request $request) {
                $orderIds = (array) $request->input('order_id', []);

                $order = Order::query()->find(Arr::first($orderIds));

                if (! $order) {
                    return [];
                }

                $event = Event::query()->whereKey($order->event_id)->first();

                return [
                    'amount' => (float) $order->total,
                    'currency' => strtoupper(get_application_currency()->title),
                    'order_id' => $orderIds,
                    'description' => trans('plugins/payment::payment.payment_description', ['order_id' => Arr::first($orderIds), 'site_url' => request()->getHost()]),
                    'return_url' => $request->input('return_url'),
                    'callback_url' => $request->input('callback_url'),
                    'products' => [
                        [
                            'name' => $event->name,
                            'image' => $event->images,
                            'id' => $event->id,
                            'qty' => $order->quantity,
                            'price_per_order' => $order->subtotal,
                        ],
                    ],
                    'customer' => $order->account ?? [],
                    'orders' => [$order],
                    'checkout_token' => session('checkout_token'),
                    'customer_id' => auth('account')->check() ? auth('account')->id() : null,
                    'customer_type' => Account::class,
                    'address' => [
                        'name' => $order->account_name,
                        'email' => $order->account_email,
                        'phone' => $order->account ? $order->account->phone : null,
                        'address' => null,
                        'country' => null,
                    ],
                ];
            }, 120, 2);
        }

        if (defined('PAYMENT_FILTER_PAYMENT_INFO_DETAIL')) {
            add_filter(PAYMENT_FILTER_PAYMENT_INFO_DETAIL, function ($html, $payment) {
                if (! $payment->order_id) {
                    return $html;
                }

                $order = Order::query()->find($payment->order_id);

                if (! $order) {
                    return $html;
                }

                return view('plugins/events-booking::partials.payment-info', compact('order'))->render() . $html;
            }, 123, 2);
        }

        if (defined('ACTION_AFTER_UPDATE_PAYMENT')) {
            add_action(ACTION_AFTER_UPDATE_PAYMENT, function ($request, $payment) {
                if (
                    in_array($payment->payment_channel, [PaymentMethodEnum::COD, PaymentMethodEnum::BANK_TRANSFER])
                    && $request->input('status') == PaymentStatusEnum::COMPLETED
                ) {
                    $order = Order::query()->find($payment->order_id);

                    $order->status = OrderStatusEnum::COMPLETED;
                    $order->completed_at = Carbon::now();
                    $order->save();

                    OrderCompleted::dispatch($order);
                }
            }, 123, 2);
        }

        if (defined('PAGE_MODULE_SCREEN_NAME')) {
            add_filter(PAGE_FILTER_FRONT_PAGE_CONTENT, [$this, 'renderEventsPage'], 2, 2);
        }

        $this->app['events']->listen(RenderingThemeOptionSettings::class, function (): void {
            add_action(RENDERING_THEME_OPTIONS_PAGE, [$this, 'addThemeOptions'], 35);
        });

        PageTable::beforeRendering(function (): void {
            add_filter(PAGE_FILTER_PAGE_NAME_IN_ADMIN_LIST, [$this, 'addAdditionNameToPageName'], 147, 2);
        });

        add_filter('cms_twig_compiler', function (TwigCompiler $twigCompiler) {
            if (! array_key_exists(TwigExtension::class, $twigCompiler->getExtensions())) {
                $twigCompiler->addExtension(new TwigExtension());
            }

            return $twigCompiler;
        }, 123);
    }

    public function addThemeOptions(): void
    {
        $pages = Page::query()
            ->wherePublished()
            ->pluck('name', 'id')
            ->all();

        theme_option()
            ->setSection([
                'title' => trans('plugins/events-booking::settings.name'),
                'id' => 'opt-text-subsection-events-booking',
                'subsection' => true,
                'icon' => 'ti ti-calendar-check',
                'fields' => [
                    [
                        'id' => 'events_page_id',
                        'type' => 'customSelect',
                        'label' => trans('plugins/events-booking::settings.theme_options.events_page_id'),
                        'attributes' => [
                            'name' => 'events_page_id',
                            'list' => [0 => trans('plugins/events-booking::settings.theme_options.select')] + $pages,
                            'value' => '',
                            'options' => [
                                'class' => 'form-control',
                            ],
                        ],
                    ],
                    [
                        'id' => 'number_of_events_per_page',
                        'type' => 'number',
                        'label' => trans('plugins/events-booking::settings.theme_options.number_of_events_per_page'),
                        'attributes' => [
                            'name' => 'number_of_events_per_page',
                            'value' => 12,
                            'options' => [
                                'class' => 'form-control',
                            ],
                        ],
                    ],
                ],
            ]);
    }

    public function renderEventsPage(?string $content, Page $page): ?string
    {
        if ($page->getKey() == $this->getEventsPageId()) {
            $view = 'plugins/events-booking::themes.events';

            if (view()->exists($viewPath = Theme::getThemeNamespace() . '::views.events-booking.events')) {
                $view = $viewPath;
            }

            return view($view, [
                'events' => get_all_events(theme_option('number_of_events_per_page', 12)),
            ])->render();
        }

        return $content;
    }

    public function addAdditionNameToPageName(?string $name, Page $page): ?string
    {
        if ($page->getKey() == $this->getEventsPageId()) {
            $subTitle = Html::tag('span', trans('plugins/events-booking::settings.events_page'), ['class' => 'additional-page-name'])
                ->toHtml();

            if (Str::contains($name, ' —')) {
                return $name . ', ' . $subTitle;
            }

            return $name . ' —' . $subTitle;
        }

        return $name;
    }

    protected function getEventsPageId(): int|string|null
    {
        return theme_option('events_page_id');
    }
}
