<?php

namespace ArchiElite\EventsBooking\Supports;

use Botble\Base\Supports\Helper;
use Botble\Location\Models\Country;
use Botble\Location\Models\State;
use Botble\Theme\Facades\Theme;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;

class EventsBookingHelper
{
    public function getLoginOption(): string
    {
        return get_events_booking_setting('login_option', 'email');
    }

    public function isLoginUsingPhone(): bool
    {
        return $this->getLoginOption() == 'phone';
    }

    public function isEnableEmailVerification(): bool
    {
        return (bool) get_events_booking_setting('verify_customer_email', 0);
    }

    public function viewPath(string $view): string
    {
        $themeView = Theme::getThemeNamespace() . '::views.events-booking.' . $view;

        if (view()->exists($themeView)) {
            return $themeView;
        }

        return 'plugins/events-booking::themes.' . $view;
    }

    public function getAvailableCountries(): array
    {
        if (! empty($this->availableCountries)) {
            return $this->availableCountries;
        }

        $countries = ['' => __('Select country...')];

        if ($this->loadCountriesStatesCitiesFromPluginLocation()) {
            $selectedCountries = Country::query()
                ->wherePublished()
                ->orderBy('order')
                ->orderBy('name')
                ->select('name', 'id')
                ->get()
                ->mapWithKeys(fn (Country $item) => [$item->getKey() => $item->name])
                ->all();

            if (! empty($selectedCountries)) {
                $this->availableCountries = $countries + $selectedCountries;

                return $this->availableCountries;
            }
        }

        try {
            $selectedCountries = json_decode(get_events_booking_setting('available_countries'), true);
        } catch (Exception) {
            $selectedCountries = [];
        }

        if (empty($selectedCountries)) {
            $this->availableCountries = $countries + Helper::countries();

            return $this->availableCountries;
        }

        foreach (Helper::countries() as $key => $item) {
            if (in_array($key, $selectedCountries)) {
                $countries[$key] = $item;
            }
        }
        $this->availableCountries = $countries;

        return $this->availableCountries;
    }

    public function loadCountriesStatesCitiesFromPluginLocation(): bool
    {
        if (isset($this->loadLocationDataFromPluginLocation)) {
            return $this->loadLocationDataFromPluginLocation;
        }

        if (
            ! is_plugin_active('location')
            || ! Country::query()->exists()
            || ! State::query()->exists()
        ) {
            $this->loadLocationDataFromPluginLocation = false;

            return false;
        }

        $this->loadLocationDataFromPluginLocation = (bool) get_events_booking_setting('load_countries_states_cities_from_location_plugin', 0);

        return $this->loadLocationDataFromPluginLocation;
    }

    public function isZipCodeEnabled(): bool
    {
        return (bool) get_events_booking_setting('zip_code_enabled', '0');
    }

    public function disableOrderInvoiceUntilOrderConfirmed(): bool
    {
        return (bool) get_events_booking_setting('disable_order_invoice_until_order_confirmed', 0);
    }

    public function getCountryNameById(int|string|null $countryId): ?string
    {
        if (! $countryId) {
            return null;
        }

        if ($this->loadCountriesStatesCitiesFromPluginLocation()) {
            $countryName = Country::query()
                ->where('id', $countryId)
                ->value('name');

            if (! empty($countryName)) {
                return $countryName;
            }
        }

        return Helper::getCountryNameByCode($countryId);
    }

    public function getDateRangeInReport(Request $request): array
    {
        $startDate = Carbon::now()->subDays(29);
        $endDate = Carbon::now();

        if ($request->input('date_from')) {
            try {
                $startDate = Carbon::now()->createFromFormat('Y-m-d', $request->input('date_from'));
            } catch (Exception) {
                $startDate = Carbon::now()->subDays(29);
            }
        }

        if ($request->input('date_to')) {
            try {
                $endDate = Carbon::now()->createFromFormat('Y-m-d', $request->input('date_to'));
            } catch (Exception) {
                $endDate = Carbon::now();
            }
        }

        if ($endDate->gt(Carbon::now())) {
            $endDate = Carbon::now();
        }

        if ($startDate->gt($endDate)) {
            $startDate = Carbon::now()->subDays(29);
        }

        $predefinedRange = $request->input('predefined_range', trans('plugins/events-booking::booking-report.ranges.last_30_days'));

        return [$startDate, $endDate, $predefinedRange];
    }
}
