<?php

namespace ArchiElite\EventsBooking\Supports;

use ArchiElite\EventsBooking\Events\OrderPaymentConfirmedEvent;
use ArchiElite\EventsBooking\Models\Order;
use Botble\Base\Facades\BaseHelper;
use Botble\Base\Facades\EmailHandler;
use Botble\Base\Supports\EmailHandler as EmailHandlerSupport;
use Botble\Media\Facades\RvMedia;
use Botble\Payment\Enums\PaymentStatusEnum;
use Illuminate\Support\Facades\Auth;

class OrderHelper
{
    public function setEmailVariables(Order $order): EmailHandlerSupport
    {
        return EmailHandler::setModule(EVENTS_BOOKING_MODULE_SCREEN_NAME)
            ->setVariableValues($this->getEmailVariables($order));
    }

    public function getEmailVariables(Order $order)
    {
        $paymentMethod = '&mdash;';

        if (is_plugin_active('payment')) {
            $paymentMethod = $order->payment->payment_channel->label();
        }

        $order->loadMissing(['event', 'account']);

        $event = $order->event;
        $account = $order->account;

        return apply_filters('events_booking_order_email_variables', [
            'company_name' => get_events_booking_setting('company_name_for_invoicing'),
            'company_address' => get_events_booking_setting('company_address_for_invoicing'),
            'order_id' => $order->code,
            'order_code' => $order->code,
            'order_note' => $order->description,
            'customer_name' => BaseHelper::clean($order->account_name),
            'customer_email' => $order->account_email,
            'customer_phone' => $account ? $account->phone : null,
            'ticket_list' => view('plugins/events-booking::emails.partials.ticket-list', compact('order'))
                ->render(),
            'payment_method' => $paymentMethod,
            'order' => $order->toArray(),
            'tickets' => $order->tickets->toArray(),
            'event_name' => $event->name,
            'event_image' => RvMedia::getImageUrl($event->image, 'medium'),
            'event_location' => $event->address,
            'start_date' => $event->started_at,
            'order_completed_title' => trans('plugins/events-booking::events-booking.order_confirm.order_completed_title'),
            'order_dear' => trans('plugins/events-booking::events-booking.order_confirm.order_dear'),
            'order_your_tickets' => trans('plugins/events-booking::events-booking.order_confirm.order_your_tickets'),
        ], $order);
    }

    public function sendOrderConfirmationEmail(Order $order): bool
    {
        $mailer = EmailHandler::setModule(EVENTS_BOOKING_MODULE_SCREEN_NAME);

        if ($mailer->templateEnabled('account_new_order')) {
            $this->setEmailVariables($order);

            EmailHandler::send(
                $mailer->getTemplateContent('account_new_order'),
                $mailer->getTemplateSubject('account_new_order'),
                $order->account ? $order->account->email : $order->account_email,
            );
        }

        return true;
    }

    public function sendOrderCompletedEmail(Order $order): bool
    {
        $mailer = EmailHandler::setModule(EVENTS_BOOKING_MODULE_SCREEN_NAME);

        if ($mailer->templateEnabled('order_completed')) {
            $this->setEmailVariables($order);

            EmailHandler::send(
                $mailer->getTemplateContent('order_completed'),
                $mailer->getTemplateSubject('order_completed'),
                $order->account ? $order->account->email : $order->account_email,
            );
        }

        return true;
    }

    public function confirmPayment(Order $order): bool
    {
        if (! is_plugin_active('payment')) {
            return false;
        }

        $payment = $order->payment;

        if (! $payment) {
            return false;
        }

        $payment->status = PaymentStatusEnum::COMPLETED;
        $payment->amount = $payment->amount ?: 0;
        $payment->user_id = Auth::id();
        $payment->save();

        event(new OrderPaymentConfirmedEvent($order, Auth::user()));

        $mailer = EmailHandler::setModule(EVENTS_BOOKING_MODULE_SCREEN_NAME);
        if ($mailer->templateEnabled('order_confirm_payment')) {
            $this->setEmailVariables($order);
            $mailer->sendUsingTemplate(
                'order_confirm_payment',
                $order->account ? $order->account->email : $order->account_email,
            );
        }

        return true;
    }
}
