<?php

namespace ArchiElite\EventsBooking\Tables;

use ArchiElite\EventsBooking\Models\Event;
use Botble\Base\Facades\BaseHelper;
use Botble\Base\Facades\Html;
use Botble\Table\Abstracts\TableAbstract;
use Botble\Table\Actions\Action;
use Botble\Table\Actions\DeleteAction;
use Botble\Table\Actions\EditAction;
use Botble\Table\BulkActions\DeleteBulkAction;
use Botble\Table\BulkChanges\CreatedAtBulkChange;
use Botble\Table\BulkChanges\NameBulkChange;
use Botble\Table\Columns\Column;
use Botble\Table\Columns\CreatedAtColumn;
use Botble\Table\Columns\FormattedColumn;
use Botble\Table\Columns\IdColumn;
use Botble\Table\Columns\LinkableColumn;
use Botble\Table\Columns\NameColumn;
use Botble\Table\Columns\StatusColumn;
use Botble\Table\HeaderActions\CreateHeaderAction;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;

class EventTable extends TableAbstract
{
    public function setup(): void
    {
        $this
            ->model(Event::class)
            ->addHeaderAction(CreateHeaderAction::make()->route('events-booking.events.create'))
            ->addActions([
                Action::make('analytics')
                    ->route('events-booking.events.analytics')
                    ->label(trans('plugins/events-booking::events-booking.event.analytics.name'))
                    ->icon('ti ti-chart-line')
                    ->color('info'),
                EditAction::make()->route('events-booking.events.edit'),
                DeleteAction::make()->route('events-booking.events.destroy'),
            ])
            ->addColumns([
                IdColumn::make(),
                NameColumn::make('name')->route('events-booking.events.edit')
                    ->title(trans('core/base::tables.name')),
                Column::make('address')
                    ->title(trans('plugins/events-booking::events-booking.event.form.address')),
                LinkableColumn::make('account_id')
                    ->urlUsing(function (LinkableColumn $column) {
                        $event = $column->getItem();

                        if (! $event->account_id) {
                            return null;
                        }

                        return route('events-booking.accounts.edit', $event->account_id);
                    })
                    ->label(trans('plugins/events-booking::events-booking.event.form.organizer'))
                    ->getValueUsing(function (FormattedColumn $column) {
                        $item = $column->getItem();
                        $item->loadMissing('organizer');

                        if (! $organizer = $item->organizer) {
                            return '-';
                        }

                        return $organizer->name;
                    }),
                Column::make('booked_seats')
                    ->searchable(false)
                    ->sortable(false)
                    ->title(trans('plugins/events-booking::events-booking.event.form.booked_seats')),
                Column::make('seats')
                    ->title(trans('plugins/events-booking::events-booking.event.form.seats')),
                FormattedColumn::make('booking_closed_at')
                    ->getValueUsing(function (FormattedColumn $column) {
                        $item = $column->getItem();

                        if (! $item->booking_closed_at) {
                            return '∞';
                        }

                        $attributes = [];

                        if (Carbon::now()->greaterThan($item->booking_closed_at)) {
                            $attributes['class'] = 'text-danger';
                        }

                        return Html::tag('span', BaseHelper::formatDateTime($item->booking_closed_at), $attributes);
                    })
                    ->title(trans('plugins/events-booking::events-booking.event.form.booking_close_at')),
                StatusColumn::make(),
                CreatedAtColumn::make(),
            ])
            ->addBulkActions([
                DeleteBulkAction::make()->permission('events-booking.events.destroy'),
            ])
            ->addBulkChanges([
                NameBulkChange::make(),
                CreatedAtBulkChange::make(),
            ])
            ->queryUsing(function (Builder $query) {
                $query->select([
                    'id',
                    'name',
                    'account_id',
                    'address',
                    'seats',
                    'status',
                    'booking_closed_at',
                    'created_at',
                ]);
            });
    }
}
