<?php

namespace Database\Seeders\Themes\Main;

use ArchiElite\EventsBooking\Enums\AccountStatusEnum;
use ArchiElite\EventsBooking\Enums\AccountTypeEnum;
use ArchiElite\EventsBooking\Enums\EventStatusEnum;
use ArchiElite\EventsBooking\Enums\OrderStatusEnum;
use ArchiElite\EventsBooking\Enums\TicketStatusEnum;
use ArchiElite\EventsBooking\Facades\InvoiceHelper;
use ArchiElite\EventsBooking\Models\Account;
use ArchiElite\EventsBooking\Models\Event;
use ArchiElite\EventsBooking\Models\Order;
use ArchiElite\EventsBooking\Models\Ticket;
use Botble\Location\Models\City;
use Botble\Location\Models\Country;
use Botble\Location\Models\State;
use Botble\Payment\Enums\PaymentMethodEnum;
use Botble\Payment\Enums\PaymentStatusEnum;
use Botble\Payment\Models\Payment;
use Botble\Slug\Facades\SlugHelper;
use Botble\Slug\Models\Slug;
use Botble\Theme\Database\Seeders\ThemeSeeder;
use Carbon\Carbon;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class EventSeeder extends ThemeSeeder
{
    public function run(): void
    {
        Event::query()->truncate();
        Order::query()->truncate();
        Ticket::query()->truncate();

        $this->uploadFiles('events', 'main');

        $faker = $this->fake();

        $accounts = Account::query()->where('type', AccountTypeEnum::ORGANIZER)->pluck('id')->all();

        $speakerIds = Account::query()->where('type', AccountTypeEnum::SPEAKER)->pluck('id')->all();

        $events = [
            [
                'name' => 'CES (Consumer Electronics Show)',
                'description' => 'The world\'s largest consumer electronics exhibition, showcasing the latest products and technologies in consumer electronics.',
            ],
            [
                'name' => 'Mobile World Congress',
                'description' => 'The world\'s leading mobile technology event, displaying the latest trends and innovations in mobile technology and telecommunications.',
            ],
            [
                'name' => 'Google I/O',
                'description' => 'Google\'s annual developer conference, announcing new products, services, and technologies from Google.',
            ],
            [
                'name' => 'Apple WWDC',
                'description' => 'Apple\'s Worldwide Developers Conference, introducing software updates and occasionally new hardware.',
            ],
            [
                'name' => 'AWS re:Invent',
                'description' => 'Amazon Web Services\' annual conference, focusing on cloud computing and new AWS services.',
            ],
            [
                'name' => 'Microsoft Build',
                'description' => 'Microsoft\'s developer event, introducing new development tools and platforms.',
            ],
            [
                'name' => 'Facebook F8',
                'description' => 'Meta\'s (Facebook) developer conference, focusing on the future of social media and virtual reality technology.',
            ],
            [
                'name' => 'GitHub Universe',
                'description' => 'GitHub\'s annual event, discussing the future of software development and open-source collaboration.',
            ],
            [
                'name' => 'Game Developers Conference',
                'description' => 'The largest conference in the gaming industry, focusing on game development and gaming technology.',
            ],
            [
                'name' => 'Web Summit',
                'description' => 'One of the world\'s largest technology conferences, focusing on technology trends and startups.',
            ],
        ];

        foreach ($events as $index => $item) {
            $event = Event::query()->create([
                'name' => Arr::get($item, 'name'),
                'unique_id' => $faker->uuid,
                'account_id' => $accounts[rand(0, count($accounts) - 1)],
                'description' => Arr::get($item, 'description'),
                'content' => file_get_contents(database_path('seeders/contents/event.html')),
                'image' => $this->filePath('events/' . $index + 1 . '.webp', 'main'),
                'price' => $faker->randomFloat(2, 0, 100),
                'short_description' => $faker->sentence,
                'address' => $faker->address,
                'country_id' => $countryId = Country::query()->inRandomOrder()->first()->getKey(),
                'state_id' => $stateId = State::query()->where('country_id', $countryId)->inRandomOrder()->first()?->getKey(),
                'city_id' => $stateId ? City::query()->where('state_id', $stateId)->inRandomOrder()->first()?->getKey() : null,
                'latitude' => $faker->latitude,
                'longitude' => $faker->longitude,
                'started_at' => Carbon::now()->addDays(floor($index / 4))->setTime(9, 00, 00),
                'ended_at' => Carbon::now()->addDays(rand(10, 20))->setTime(18, 00, 00),
                'seats' => $faker->numberBetween(50, 100),
                'status' => EventStatusEnum::APPROVED,
                'is_featured' => $faker->boolean,
            ]);

            Slug::query()->create([
                'reference_type' => Event::class,
                'reference_id' => $event->getKey(),
                'key' => Str::slug($event->name),
                'prefix' => SlugHelper::getPrefix(Event::class),
            ]);

            $speakerData = [];

            for ($i = 0; $i <= 3; $i++) {
                $speakerData[] = [
                    'priority' => $i,
                    'event_id' => $event->id,
                    'speaker_id' => $speakerIds[rand(1, count($speakerIds) - 1)],
                    'title' => $faker->text(100),
                ];
            }

            $uniqueDataSpeakers = collect($speakerData)->unique('speaker_id')->all();

            $event->speakers()->sync($uniqueDataSpeakers);
        }

        $events = Event::query()
            ->where('status', EventStatusEnum::APPROVED)
            ->limit(5)
            ->get();

        foreach ($events as $event) {
            for ($i = 0; $i < 10; $i++) {
                $slots = rand(1, 5);
                $account = Account::query()->where('type', AccountTypeEnum::ATTENDEE)->inRandomOrder()->first();
                $order = new Order();

                $order->token = Str::uuid();
                $order->price = $event->price ?? 0;
                $order->subtotal = $slots * $event->price;
                $order->total = $slots * $event->price;
                $order->quantity = $slots;
                $order->note = $faker->realText(100);
                $order->account_name = $account->name;
                $order->account_email = $account->email;
                $order->account_id = $account->getKey();
                $order->event_id = $event->getKey();
                $order->currency_id = get_application_currency()->getKey();
                $order->code = Str::upper(Str::random(12));
                $order->event_name = $event->name;
                $order->event_ticket_price = $event->price;
                $order->status = $faker->randomElement(OrderStatusEnum::values());

                $order->save();

                if (is_plugin_active('payment')) {
                    $payment = new Payment();
                    $payment->charge_id = Str::upper(Str::random(12));
                    $payment->order_id = $order->getKey();
                    $payment->amount = $order->total;
                    $payment->status = PaymentStatusEnum::COMPLETED;
                    $payment->payment_channel = PaymentMethodEnum::BANK_TRANSFER;
                    $payment->currency = get_application_currency()->getKey();
                    $payment->save();

                    $order->payment_id = $payment->getKey();
                    $order->save();
                }

                $ticketsData = [];

                for ($j = 0; $j <= $slots; $j++) {
                    $ticketsData[] = [
                        'order_id' => $order->getKey(),
                        'name' => $faker->name,
                        'email' => $faker->email,
                        'phone' => '0' . $faker->numerify('#########'),
                        'note' => $faker->realText(100),
                        'code' => Str::upper(Str::random(12)),
                        'created_at' => $now = Carbon::now(),
                        'updated_at' => $now,
                        'status' => $faker->randomElement(TicketStatusEnum::values()),
                    ];
                }

                Ticket::query()->insert($ticketsData);

                InvoiceHelper::store($order);
            }
        }

        $orders = Order::query()->with('event')->where('status', OrderStatusEnum::COMPLETED)->get();

        foreach ($orders as $order) {
            $event = $order->event;

            $tickets = Ticket::query()
                ->where('order_id', $order->getKey())
                ->get();

            foreach ($tickets as $ticket) {
                $account = Account::query()->where('email', $ticket->email)->first();

                if (! $account) {
                    $account = Account::query()->create([
                        'email' => $ticket->email,
                        'name' => $ticket->name,
                        'phone' => $ticket->phone,
                        'status' => AccountStatusEnum::ACTIVATED,
                        'type' => AccountTypeEnum::ATTENDEE,
                        'password' => Hash::make('password'),
                    ]);
                }

                $event->attendees()->attach($account->getKey(), [
                    'hash' => $ticket->code,
                ]);
            }
        }
    }
}
