<?php

namespace ArchiElite\EventsBooking\Exporters;

use ArchiElite\EventsBooking\Enums\AccountTypeEnum;
use ArchiElite\EventsBooking\Models\Account;
use ArchiElite\EventsBooking\Models\Event;
use Botble\Base\Facades\Assets;
use Botble\DataSynchronize\Exporter\ExportColumn;
use Botble\DataSynchronize\Exporter\ExportCounter;
use Botble\DataSynchronize\Exporter\Exporter;
use Illuminate\Contracts\View\View;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;

class AttendeeExporter extends Exporter
{
    protected ?Event $event = null;

    public function getLabel(): string
    {
        return trans('plugins/events-booking::events-booking.export.attendee.title');
    }

    public function columns(): array
    {
        return [
            ExportColumn::make('event_id')->label(trans('plugins/events-booking::events-booking.account.form.event_id'))->disabled(),
            ExportColumn::make('id')->label('ID'),
            ExportColumn::make('name')->label(trans('plugins/events-booking::events-booking.account.form.name')),
            ExportColumn::make('email')->label(trans('plugins/events-booking::events-booking.account.form.email')),
            ExportColumn::make('phone')->label(trans('plugins/events-booking::events-booking.account.form.phone')),
            ExportColumn::make('dob')->label(trans('plugins/events-booking::events-booking.account.form.dob')),
            ExportColumn::make('created_at'),

        ];
    }

    public function counters(): array
    {
        $counter = 0;

        if (request()->input('event_id')) {
            $event = Event::query()
                ->withCount('attendees')
                ->find(request()->input('event_id'));

            $counter = $event ? $event->attendees_count : 0;
        }

        return [
            ExportCounter::make()
                ->label(trans('plugins/events-booking::events-booking.export.attendee.total_attendees'))
                ->value($counter),
        ];
    }

    public function collection(): Collection
    {
        $this->validate();

        return Account::query()
            ->whereHas('events', function (Builder $query) {
                $query->where('event_id', $this->event->id);
            })
            ->select('id', 'name', 'email', 'phone', 'dob', 'created_at', 'type')
            ->where('type', AccountTypeEnum::ATTENDEE)
            ->get()
            ->transform(fn (Account $item) => [
                'event_id' => $this->event->id,
                ...$item->toArray(),
                'dob' => $item->dob ? $item->dob->format('Y-m-d') : null,
                'created_at' => $item->created_at->format('Y-m-d H:i:s'),
            ]);
    }

    public function render(): View
    {
        Assets::addScriptsDirectly('vendor/core/packages/data-synchronize/js/data-synchronize.js');
        Assets::addScriptsDirectly('vendor/core/plugins/events-booking/js/export-attendees.js');

        $defaultEvent = null;

        if (request()->input('event_id')) {
            $defaultEvent = Event::query()
                ->withCount('attendees')
                ->find(request()->input('event_id'));
        }

        return view('plugins/events-booking::exports.event', [
            'exporter' => $this,
            'event' => $defaultEvent,
        ]);
    }

    protected function validate(): void
    {
        $request = request();

        if (! $eventId = $request->input('event_id')) {
            throw new \Exception(trans('plugins/events-booking::events-booking.export.attendee.errors.required_event'));
        }

        $event = Event::query()->find($eventId);

        if (! $event) {
            throw new \Exception(trans('plugins/events-booking::events-booking.export.attendee.errors.event_not_found'));
        }

        $this->event = $event;
    }
}
