<?php

namespace ArchiElite\EventsBooking\Http\Controllers;

use ArchiElite\EventsBooking\Enums\OrderStatusEnum;
use ArchiElite\EventsBooking\Events\TicketResendMailEvent;
use ArchiElite\EventsBooking\Facades\OrderHelper;
use ArchiElite\EventsBooking\Http\Requests\UpdateOrderRequest;
use ArchiElite\EventsBooking\Models\Order;
use ArchiElite\EventsBooking\Models\Ticket;
use ArchiElite\EventsBooking\Tables\OrderTable;
use Botble\Base\Facades\Assets;
use Botble\Base\Http\Actions\DeleteResourceAction;
use Botble\Base\Http\Controllers\BaseController;

class OrderController extends BaseController
{
    public function __construct()
    {
        $this
            ->breadcrumb()
            ->add(trans('plugins/events-booking::events-booking.name'))
            ->add(trans('plugins/events-booking::events-booking.order.name'), route('events-booking.orders.index'));
    }

    public function index(OrderTable $table)
    {
        $this->pageTitle(trans('plugins/events-booking::events-booking.order.name'));

        return $table->renderTable();
    }

    public function edit(Order $order)
    {
        $this->pageTitle(trans('core/base::forms.edit_item', ['name' => $order->code]));

        Assets::addScriptsDirectly([
            'vendor/core/plugins/events-booking/js/order.js',
        ]);

        $orderStatuses = OrderStatusEnum::labels();

        $order->loadMissing('tickets');

        return view('plugins/events-booking::orders.edit', compact('order', 'orderStatuses'));
    }

    public function update(Order $order, UpdateOrderRequest $request)
    {
        $order->update($request->validated());

        return $this
            ->httpResponse()
            ->setPreviousUrl(route('events-booking.orders.index'))
            ->withUpdatedSuccessMessage();
    }

    public function destroy(Order $order)
    {
        return DeleteResourceAction::make($order);
    }

    public function postConfirmPayment(Order $order)
    {
        if ($order->status == OrderStatusEnum::PENDING) {
            $order->status = OrderStatusEnum::PROCESSING;
        }

        $order->save();

        $order->load(['payment']);

        OrderHelper::confirmPayment($order);

        return $this
            ->httpResponse()
            ->setMessage(trans('plugins/events-booking::events-booking.order.confirm_payment_success'));
    }

    public function resendMail(string $orderId, string $ticketId)
    {
        $order = Order::query()->where('id', $orderId)->firstOrFail();
        $ticket = Ticket::query()->where('id', $ticketId)->firstOrFail();

        if (! $order || ! $ticket) {
            return $this
                ->httpResponse()
                ->setError()
                ->setMessage(trans('plugins/events-booking::events-booking.order.resend_mail_failed'));
        }

        if (! $order->isCompleted()) {
            return $this
                ->httpResponse()
                ->setError()
                ->setMessage(trans('plugins/events-booking::events-booking.order.resend_mail_failed'));
        }

        event(new TicketResendMailEvent($order, $ticket));

        return $this
            ->httpResponse()
            ->setMessage(trans('plugins/events-booking::events-booking.order.resend_mail_success'));
    }
}
