<?php

namespace ArchiElite\EventsBooking\Models;

use ArchiElite\EventsBooking\Enums\InvoiceStatusEnum;
use Botble\Base\Models\BaseModel;
use Botble\Payment\Models\Payment;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;

class Invoice extends BaseModel
{
    protected $table = 'eb_invoices';

    protected $fillable = [
        'code',
        'reference_id',
        'reference_type',
        'event_name',
        'event_ticket_price',
        'account_name',
        'company_name',
        'company_logo',
        'account_email',
        'account_phone',
        'sub_total',
        'amount',
        'tax_amount',
        'discount_amount',
        'payment_id',
        'status',
        'paid_at',
        'description',
    ];

    protected $casts = [
        'sub_total' => 'float',
        'amount' => 'float',
        'tax_amount' => 'float',
        'discount_amount' => 'float',
        'status' => InvoiceStatusEnum::class,
        'paid_at' => 'datetime',
    ];

    protected static function booted(): void
    {
        static::creating(function (Invoice $invoice) {
            $invoice->code = static::generateUniqueCode();
        });

        static::deleted(function (Invoice $invoice) {
            $invoice->items()->delete();
        });
    }

    public function reference(): MorphTo
    {
        return $this->morphTo();
    }

    public function items(): HasMany
    {
        return $this->hasMany(InvoiceItem::class);
    }

    public function payment(): BelongsTo
    {
        return $this->belongsTo(Payment::class)->withDefault();
    }

    public static function generateUniqueCode(): string
    {
        $prefix = get_events_booking_setting('invoice_code_prefix', 'INV-');
        $nextInsertId = BaseModel::determineIfUsingUuidsForId() ? static::query()->count() + 1 : static::query()->max('id') + 1;

        do {
            $code = sprintf('%s%d', $prefix, $nextInsertId);
            $nextInsertId++;
        } while (static::query()->where('code', $code)->exists());

        return $code;
    }
}
