<?php

namespace ArchiElite\EventsBooking\Providers;

use ArchiElite\EventsBooking\Http\Middleware\RedirectIfAccount;
use ArchiElite\EventsBooking\Http\Middleware\RedirectIfNotAccount;
use ArchiElite\EventsBooking\Models\Account;
use ArchiElite\EventsBooking\Models\Event;
use ArchiElite\EventsBooking\PanelSections\SettingEventsBookingPanelSection;
use Botble\Base\Facades\DashboardMenu;
use Botble\Base\Facades\EmailHandler;
use Botble\Base\Facades\PanelSectionManager;
use Botble\Base\PanelSections\PanelSectionItem;
use Botble\Base\Supports\ServiceProvider;
use Botble\Base\Traits\LoadAndPublishDataTrait;
use Botble\DataSynchronize\PanelSections\ExportPanelSection;
use Botble\DataSynchronize\PanelSections\ImportPanelSection;
use Botble\LanguageAdvanced\Supports\LanguageAdvancedManager;
use Botble\Slug\Facades\SlugHelper;
use Botble\SocialLogin\Facades\SocialService;
use Illuminate\Routing\Events\RouteMatched;
use Illuminate\Support\Facades\Route;

class EventsBookingServiceProvider extends ServiceProvider
{
    use LoadAndPublishDataTrait;

    public function register(): void
    {
        config([
            'auth.guards.account' => [
                'driver' => 'session',
                'provider' => 'accounts',
            ],
            'auth.providers.accounts' => [
                'driver' => 'eloquent',
                'model' => Account::class,
            ],
            'auth.passwords.accounts' => [
                'provider' => 'accounts',
                'table' => 'eb_account_password_resets',
                'expire' => 60,
            ],
        ]);
    }

    public function boot(): void
    {
        SlugHelper::registerModule(Event::class, 'Events');
        SlugHelper::setPrefix(Event::class, 'events');
        SlugHelper::registerModule(Account::class, 'Accounts');
        SlugHelper::setPrefix(Account::class, 'speakers');

        $this
            ->setNamespace('plugins/events-booking')
            ->loadAndPublishViews()
            ->loadHelpers()
            ->loadAndPublishConfigurations(['permissions', 'email'])
            ->loadAndPublishTranslations()
            ->publishAssets()
            ->loadRoutes()
            ->loadMigrations();

        if (defined('LANGUAGE_ADVANCED_MODULE_SCREEN_NAME')) {
            LanguageAdvancedManager::registerModule(Event::class, [
                'name',
                'short_description',
                'description',
                'content',
            ]);
        }

        DashboardMenu::default()->beforeRetrieving(function () {
            DashboardMenu::make()
                ->registerItem([
                    'id' => 'cms-plugins-events-booking',
                    'priority' => 0,
                    'parent_id' => null,
                    'name' => 'plugins/events-booking::events-booking.name',
                    'icon' => 'ti ti-box',
                ])
                ->registerItem([
                    'id' => 'cms-plugins-events-booking-booking-reports',
                    'priority' => 0,
                    'parent_id' => 'cms-plugins-events-booking',
                    'name' => 'plugins/events-booking::booking.reports',
                    'icon' => 'ti ti-chart-bar',
                    'route' => 'events-booking.booking.reports.index',
                ])
                ->registerItem([
                    'id' => 'cms-plugins-events-booking-events',
                    'priority' => 1,
                    'parent_id' => 'cms-plugins-events-booking',
                    'name' => 'plugins/events-booking::events-booking.event.name',
                    'icon' => 'ti ti-calendar-event',
                    'url' => route('events-booking.events.index'),
                    'permissions' => ['events-booking.events.index'],
                ])
                ->registerItem([
                    'id' => 'cms-plugins-events-booking-orders',
                    'priority' => 2,
                    'parent_id' => 'cms-plugins-events-booking',
                    'name' => 'plugins/events-booking::events-booking.order.name',
                    'icon' => 'ti ti-box',
                    'url' => route('events-booking.orders.index'),
                    'permissions' => ['events-booking.events.index'],
                ])
                ->registerItem([
                    'id' => 'cms-plugins-events-booking-tickets',
                    'priority' => 3,
                    'parent_id' => 'cms-plugins-events-booking',
                    'name' => 'plugins/events-booking::events-booking.ticket.name',
                    'icon' => 'ti ti-box',
                    'url' => route('events-booking.tickets.index'),
                    'permissions' => ['events-booking.events.index'],
                ])
                ->registerItem([
                    'id' => 'cms-plugins-events-booking-invoices',
                    'priority' => 4,
                    'parent_id' => 'cms-plugins-events-booking',
                    'name' => 'plugins/events-booking::invoice.name',
                    'icon' => 'ti ti-box',
                    'url' => route('events-booking.invoices.index'),
                    'permissions' => ['events-booking.events.index'],
                ])
                ->registerItem([
                    'id' => 'cms-plugins-events-booking-coupons',
                    'priority' => 5,
                    'parent_id' => 'cms-plugins-events-booking',
                    'name' => 'plugins/events-booking::coupon.name',
                    'icon' => 'ti ti-discount',
                    'url' => route('events-booking.coupons.index'),
                    'permissions' => ['events-booking.coupons.index'],
                ])
                ->registerItem([
                    'id' => 'cms-plugins-events-booking-taxes',
                    'priority' => 5,
                    'parent_id' => 'cms-plugins-events-booking',
                    'name' => 'plugins/events-booking::tax.name',
                    'icon' => 'ti ti-receipt-tax',
                    'url' => route('events-booking.taxes.index'),
                    'permissions' => ['events-booking.taxes.index'],
                ])
                ->registerItem([
                    'id' => 'cms-plugins-events-booking-accounts',
                    'priority' => 7,
                    'parent_id' => null,
                    'name' => 'plugins/events-booking::events-booking.account.name',
                    'icon' => 'ti ti-users',
                    'url' => route('events-booking.accounts.index'),
                    'permissions' => ['events-booking.accounts.index'],
                ]);
        });

        PanelSectionManager::setGroupId('data-synchronize')->beforeRendering(function (): void {
            PanelSectionManager::default()
                ->registerItem(
                    ExportPanelSection::class,
                    fn () => PanelSectionItem::make('attendees')
                        ->setTitle(trans('plugins/events-booking::events-booking.export.attendee.name'))
                        ->withDescription(trans('plugins/events-booking::events-booking.export.attendee.description'))
                        ->withPriority(999)
                        ->withPermission('events-booking.attendees.export')
                        ->withRoute('tools.data-synchronize.export.attendees.index')
                )
                ->registerItem(
                    ImportPanelSection::class,
                    fn () => PanelSectionItem::make('attendees')
                        ->setTitle(trans('plugins/events-booking::events-booking.import.attendee.name'))
                        ->withDescription(trans('plugins/events-booking::events-booking.import.attendee.description'))
                        ->withPriority(999)
                        ->withPermission('events-booking.attendees.import')
                        ->withRoute('tools.data-synchronize.import.attendees.index')
                );
        });

        DashboardMenu::for('account')->beforeRetrieving(function () {
            DashboardMenu::make()
                ->registerItem([
                    'id' => 'cms-account-overview',
                    'priority' => 10,
                    'name' => __('Overview'),
                    'url' => fn () => route('account.overview'),
                    'icon' => 'ti ti-home',
                ])
                ->registerItem([
                    'id' => 'cms-account-orders',
                    'priority' => 30,
                    'name' => __('Orders'),
                    'url' => fn () => route('account.orders'),
                    'icon' => 'ti ti-shopping-cart',
                ])
                ->registerItem([
                    'id' => 'cms-account-edit-account',
                    'priority' => 70,
                    'name' => __('Account Settings'),
                    'url' => fn () => route('account.edit-account'),
                    'icon' => 'ti ti-settings',
                ])
                ->registerItem([
                    'id' => 'cms-account-logout',
                    'priority' => 999,
                    'name' => __('Logout'),
                    'url' => fn () => route('account.logout'),
                    'icon' => 'ti ti-logout',
                ]);
        });

        PanelSectionManager::default()->beforeRendering(function () {
            PanelSectionManager::register(SettingEventsBookingPanelSection::class);
        });

        $this->app->booted(function () {
            $this->app->register(HookServiceProvider::class);

            if (defined('SOCIAL_LOGIN_MODULE_SCREEN_NAME') && Route::has('account.login')) {
                SocialService::registerModule([
                    'guard' => 'account',
                    'model' => Account::class,
                    'login_url' => route('account.login'),
                    'redirect_url' => route('account.overview'),
                ]);
            }
        });

        $this->app['events']->listen(RouteMatched::class, function () {
            $router = $this->app['router'];

            $emailConfig = config('plugins.events-booking.email', []);

            EmailHandler::addTemplateSettings('events-booking', $emailConfig);

            $router->aliasMiddleware('account', RedirectIfNotAccount::class);
            $router->aliasMiddleware('account.guest', RedirectIfAccount::class);
        });
        $this->app->register(EventServiceProvider::class);
    }
}
