<?php

namespace ArchiElite\EventsBooking\Supports;

use ArchiElite\EventsBooking\Enums\InvoiceStatusEnum;
use ArchiElite\EventsBooking\Facades\EventsBookingHelper;
use ArchiElite\EventsBooking\Models\Invoice;
use ArchiElite\EventsBooking\Models\InvoiceItem;
use ArchiElite\EventsBooking\Models\Order;
use ArchiElite\EventsBooking\Models\Ticket;
use Barryvdh\DomPDF\PDF as PDFHelper;
use Botble\Base\Facades\BaseHelper;
use Botble\Base\Supports\Pdf;
use Botble\Media\Facades\RvMedia;
use Botble\Payment\Enums\PaymentMethodEnum;
use Botble\Payment\Enums\PaymentStatusEnum;
use Botble\Payment\Models\Payment;
use Carbon\Carbon;
use Illuminate\Http\Response;

class InvoiceHelper
{
    public function store(Order $order): Invoice
    {
        $order->loadMissing('event');

        $invoiceData = [
            'reference_id' => $order->getKey(),
            'reference_type' => Order::class,
            'event_name' => $order->event_name,
            'event_ticket_price' => $order->event_ticket_price,
            'account_name' => $order->account_name,
            'company_name' => get_events_booking_setting('company_name_for_invoicing'),
            'company_logo' => get_events_booking_setting('company_logo_for_invoicing'),
            'account_email' => $order->account_email,
            'account_phone' => $order->account_phone,
            'payment_id' => null,
            'status' => InvoiceStatusEnum::COMPLETED,
            'paid_at' => Carbon::now(),
            'sub_total' => $order->subtotal,
            'amount' => $order->total,
            'tax_amount' => $order->tax_amount ?? 0,
            'discount_amount' => $order->coupon_amount ?? 0,
            'description' => $order->description,
        ];

        if (is_plugin_active('payment')) {
            $invoiceData = array_merge($invoiceData, [
                'payment_id' => $order->payment->id,
                'status' => $order->payment->status,
                'paid_at' => $order->payment->status == PaymentStatusEnum::COMPLETED ? Carbon::now() : null,
            ]);
        }

        $invoice = new Invoice($invoiceData);

        $invoice->created_at = $order->created_at;

        $invoice->save();

        $eventName = $invoice->event_name;
        $price = $order->event->price;

        foreach ($order->tickets as $ticket) {
            $invoice->items()->create([
                'reference_id' => $ticket->id,
                'reference_type' => Ticket::class,
                'name' => $eventName . ' - ' . $ticket->name . ' - ' . $ticket->email,
                'email' => $ticket->email,
                'phone' => $ticket->phone,
                'qty' => 1,
                'sub_total' => $price,
                'amount' => $price,
            ]);
        }

        return $invoice;
    }

    public function supportedDateFormats(): array
    {
        $formats = [
            'M d, Y',
            'F j, Y',
            'F d, Y',
            'Y-m-d',
            'Y-M-d',
            'd-m-Y',
            'd-M-Y',
            'm/d/Y',
            'M/d/Y',
            'd/m/Y',
            'd/M/Y',
        ];

        return apply_filters('invoice_date_formats', $formats);
    }

    public function getCompanyCountry(): ?string
    {
        return get_events_booking_setting('company_country_for_invoicing', get_events_booking_setting('store_country'));
    }

    public function getCompanyState(): ?string
    {
        return get_events_booking_setting('company_state_for_invoicing', get_events_booking_setting('store_state'));
    }

    public function getCompanyCity(): ?string
    {
        return get_events_booking_setting('company_city_for_invoicing', get_events_booking_setting('store_city'));
    }

    public function getCompanyZipCode(): ?string
    {
        return get_events_booking_setting('company_zipcode_for_invoicing', get_events_booking_setting('store_zip_code'));
    }

    public function getLanguageSupport(): string
    {
        $languageSupport = get_events_booking_setting('invoice_language_support');

        if (! empty($languageSupport)) {
            return $languageSupport;
        }

        if (get_events_booking_setting('invoice_support_arabic_language', false)) {
            return 'arabic';
        }

        if (get_events_booking_setting('invoice_support_bangladesh_language', false)) {
            return 'bangladesh';
        }

        return '';
    }

    public function getInvoiceTemplate(): string
    {
        return (new Pdf())->getContent($this->getInvoiceTemplatePath(), $this->getInvoiceTemplateCustomizedPath());
    }

    public function getInvoiceTemplatePath(): string
    {
        return plugin_path('events-booking/resources/templates/invoice.tpl');
    }

    public function getInvoiceTemplateCustomizedPath(): string
    {
        return storage_path('app/templates/events-booking/invoice.tpl');
    }

    public function getVariables(): array
    {
        return [
            'invoice.*' => trans('plugins/events-booking::invoice-template.variables.invoice_data'),
            'logo_full_path' => trans('plugins/events-booking::invoice-template.variables.site_logo'),
            'company_logo_full_path' => trans('plugins/events-booking::setting.invoice.form.company_logo'),
            'site_title' => trans('plugins/events-booking::invoice-template.variables.site_title'),
            'company_name' => trans('plugins/events-booking::invoice-template.variables.company_name'),
            'company_address' => trans('plugins/events-booking::invoice-template.variables.company_address'),
            'company_country' => trans('plugins/events-booking::invoice-template.variables.company_country'),
            'company_state' => trans('plugins/events-booking::invoice-template.variables.company_state'),
            'company_city' => trans('plugins/events-booking::invoice-template.variables.company_city'),
            'company_zipcode' => trans('plugins/events-booking::invoice-template.variables.company_zipcode'),
            'company_phone' => trans('plugins/events-booking::invoice-template.variables.company_phone'),
            'company_email' => trans('plugins/events-booking::invoice-template.variables.company_email'),
            'company_tax_id' => trans('plugins/events-booking::invoice-template.variables.company_tax_id'),
            'payment_method' => __('Payment method'),
            'payment_status' => __('Payment status'),
            'payment_description' => __('Payment description'),
        ];
    }

    public function getDataForPreview(): Invoice
    {
        $invoice = new Invoice([
            'code' => 'INV-1',
            'account_name' => 'Odie Miller',
            'store_name' => 'LinkedIn',
            'store_address' => '701 Norman Street Los Angeles California 90008',
            'account_email' => 'contact@example.com',
            'account_phone' => '+0123456789',
            'status' => InvoiceStatusEnum::PENDING,
            'created_at' => Carbon::now()->toDateTimeString(),
        ]);

        $items = [];

        foreach (range(1, 3) as $i) {
            $amount = rand(10, 1000);
            $qty = rand(1, 10);

            $items[] = new InvoiceItem([
                'name' => "Item $i",
                'sub_total' => $amount * $qty,
                'amount' => $amount,
                'qty' => $qty,
            ]);

            $invoice->amount += $amount * $qty;
            $invoice->sub_total = $invoice->amount;
        }

        $payment = new Payment([
            'payment_channel' => PaymentMethodEnum::BANK_TRANSFER,
            'status' => PaymentStatusEnum::PENDING,
        ]);

        $invoice->setRelation('payment', $payment);
        $invoice->setRelation('items', collect($items));

        return $invoice;
    }

    public function makeInvoicePDF(Invoice $invoice): PDFHelper
    {
        return (new Pdf())
            ->templatePath($this->getInvoiceTemplatePath())
            ->destinationPath($this->getInvoiceTemplateCustomizedPath())
            ->supportLanguage($this->getLanguageSupport())
            ->paperSizeA4()
            ->data($this->getDataForInvoiceTemplate($invoice))
            ->twigExtensions([
                new TwigExtension(),
            ])
            ->compile();
    }

    protected function getDataForInvoiceTemplate(Invoice $invoice): array
    {
        $logo = get_events_booking_setting('company_logo_for_invoicing') ?: (theme_option(
            'logo_in_invoices'
        ) ?: theme_option('logo'));

        $paymentDescription = null;

        if (
            is_plugin_active('payment') &&
            $invoice->payment->payment_channel == PaymentMethodEnum::BANK_TRANSFER &&
            $invoice->payment->status == PaymentStatusEnum::PENDING
        ) {
            $paymentDescription = BaseHelper::clean(
                get_payment_setting('description', $invoice->payment->payment_channel)
            );
        }

        $companyName = get_events_booking_setting('company_name_for_invoicing') ?: get_events_booking_setting('store_name');

        $companyAddress = get_events_booking_setting('company_address_for_invoicing');

        $country = EventsBookingHelper::getCountryNameById($this->getCompanyCountry());
        $state = $this->getCompanyState();
        $city = $this->getCompanyCity();

        if (! $companyAddress) {
            $companyAddress = implode(', ', array_filter([
                get_events_booking_setting('company_address_for_invoicing', get_events_booking_setting('store_address')),
                $city,
                $state,
                $country,
            ]));
        }

        $companyPhone = get_events_booking_setting('company_phone_for_invoicing') ?: get_events_booking_setting('store_phone');
        $companyEmail = get_events_booking_setting('company_email_for_invoicing') ?: get_events_booking_setting('store_email');
        $companyTaxId = get_events_booking_setting('company_tax_id_for_invoicing') ?: get_events_booking_setting(
            'store_vat_number'
        );

        $invoice->loadMissing(['items', 'reference']);

        $data = [
            'invoice' => $invoice->toArray(),
            'logo' => $logo,
            'logo_full_path' => RvMedia::getRealPath($logo),
            'site_title' => theme_option('site_title'),
            'company_logo_full_path' => RvMedia::getRealPath($logo),
            'company_name' => $companyName,
            'company_address' => $companyAddress,
            'company_country' => $country,
            'company_state' => $state,
            'company_city' => $city,
            'company_zipcode' => get_events_booking_setting('company_zipcode_for_invoicing') ?: get_events_booking_setting(
                'store_zip_code'
            ),
            'company_phone' => $companyPhone,
            'company_email' => $companyEmail,
            'company_tax_id' => $companyTaxId,
            'total_quantity' => $invoice->items->sum('qty'),
            'payment_description' => $paymentDescription,
            'settings' => [
                'using_custom_font_for_invoice' => (bool) get_events_booking_setting('using_custom_font_for_invoice'),
                'custom_font_family' => get_events_booking_setting('invoice_font_family', 'DejaVu Sans'),
                'font_family' => (int) get_events_booking_setting('using_custom_font_for_invoice', 0) == 1
                    ? get_events_booking_setting('invoice_font_family', 'DejaVu Sans')
                    : 'DejaVu Sans',
                'enable_invoice_stamp' => get_events_booking_setting('enable_invoice_stamp'),
                'date_format' => get_events_booking_setting('invoice_date_format', 'F d, Y'),
            ],
            'invoice_header_filter' => apply_filters('events_booking_invoice_header', null, $invoice),
            'invoice_body_filter' => apply_filters('events_booking_invoice_body', null, $invoice),
            'events_booking_invoice_footer' => apply_filters('events_booking_invoice_footer', null, $invoice),
            'invoice_payment_info_filter' => apply_filters('invoice_payment_info_filter', null, $invoice),
            'tax_classes_name' => $invoice->taxClassesName,
        ];

        $data['settings']['font_css'] = null;

        if ($data['settings']['using_custom_font_for_invoice'] && $data['settings']['font_family']) {
            $data['settings']['font_css'] = BaseHelper::googleFonts(
                'https://fonts.googleapis.com/css2?family=' .
                urlencode($data['settings']['font_family']) .
                ':wght@400;600;700&display=swap'
            );
        }

        $data['settings']['extra_css'] = apply_filters('events_booking_invoice_extra_css', null, $invoice);

        $data['settings']['header_html'] = apply_filters('events_booking_invoice_header_html', null, $invoice);

        if (is_plugin_active('payment')) {
            $invoice->loadMissing(['payment']);

            $data['payment_method'] = $invoice->payment->payment_channel->label();
            $data['payment_status'] = $invoice->payment->status->getValue();
            $data['payment_status_label'] = $invoice->payment->status->label();
        }

        return apply_filters('events_booking_invoice_variables', $data, $invoice);
    }

    public function streamInvoice(Invoice $invoice): Response
    {
        return $this->makeInvoicePDF($invoice)->stream();
    }

    public function downloadInvoice(Invoice $invoice): Response
    {
        return $this->makeInvoicePDF($invoice)->download(sprintf('invoice-%s.pdf', $invoice->code));
    }
}
