<?php

namespace ArchiElite\EventsBooking\Supports;

use ArchiElite\EventsBooking\Models\Order;
use ArchiElite\EventsBooking\Models\Ticket;
use ArchiElite\EventsBooking\Services\QrCodeService;
use Botble\Base\Facades\BaseHelper;
use Botble\Base\Facades\EmailHandler;
use Botble\Base\Supports\EmailHandler as EmailHandlerSupport;
use Botble\Media\Facades\RvMedia;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

class TicketHelper
{
    public function setEmailVariables(Order $order, Ticket $ticket): EmailHandlerSupport
    {
        return EmailHandler::setModule(EVENTS_BOOKING_MODULE_SCREEN_NAME)
            ->setVariableValues($this->getEmailVariables($order, $ticket));
    }

    public function sendTicketsEmail(Order $order): bool
    {
        $mailer = EmailHandler::setModule(EVENTS_BOOKING_MODULE_SCREEN_NAME);

        $enabledSendTicketEmail = $mailer->templateEnabled('event_tickets_delivery');

        if (! $enabledSendTicketEmail) {
            return false;
        }

        $order->loadMissing('tickets');

        foreach ($order->tickets as $ticket) {
            sleep(1);

            $this->sendTicketEmail($order, $ticket);
        }

        return true;
    }

    public function sendTicketEmail(Order $order, Ticket $ticket): void
    {
        $mailer = EmailHandler::setModule(EVENTS_BOOKING_MODULE_SCREEN_NAME);

        if ($mailer->templateEnabled('event_tickets_delivery') && $ticket->email) {
            $this->setEmailVariables($order, $ticket);

            EmailHandler::send(
                $mailer->getTemplateContent('event_tickets_delivery'),
                $mailer->getTemplateSubject('event_tickets_delivery'),
                $ticket->email,
                [],
                true
            );
        }
    }

    public function getEmailVariables(Order $order, Ticket $ticket)
    {
        $event = $order->event;

        return apply_filters('events_booking_order_email_variables', [
            'company_name' => get_events_booking_setting('company_name_for_invoicing'),
            'company_address' => get_events_booking_setting('company_address_for_invoicing'),
            'order_id' => $order->code,
            'ticket_qrcode' => $this->getQrCodeImage($ticket),
            'customer_name' => BaseHelper::clean($ticket->name),
            'customer_email' => $ticket->email,
            'customer_phone' => $ticket->phone,
            'order_title' => trans('plugins/events-booking::events-booking.order_confirm.order_title', ['name' => BaseHelper::clean($event->name)]),
            'event_image' => RvMedia::getImageUrl($event->image, 'medium'),
            'event_name' => BaseHelper::clean($event->name),
            'event_location' => $event->address,
            'start_date' => $event->started_at,
            'ticket_thanks' => trans('plugins/events-booking::events-booking.event_tickets_delivery.ticket_thanks'),
            'ticket_dear' => trans('plugins/events-booking::events-booking.event_tickets_delivery.ticket_dear'),
            'ticket_instruction' => trans('plugins/events-booking::events-booking.event_tickets_delivery.ticket_instruction'),
        ], $ticket);
    }

    public function getQrCodeImage(Ticket $ticket): string
    {
        $path = '/qrcode/' . $ticket->code . '.png';

        if (! File::exists(Storage::path($path))) {
            app(QrCodeService::class)->saveToFile($ticket->code, '/qrcode/' . $ticket->code . '.png');
        }

        return Storage::url($path);
    }
}
