<?php

namespace ArchiElite\EventsBooking\Tables;

use ArchiElite\EventsBooking\Models\Order;
use Botble\Table\Abstracts\TableAbstract;
use Botble\Table\Actions\DeleteAction;
use Botble\Table\Actions\EditAction;
use Botble\Table\BulkActions\DeleteBulkAction;
use Botble\Table\BulkChanges\CreatedAtBulkChange;
use Botble\Table\Columns\Column;
use Botble\Table\Columns\CreatedAtColumn;
use Botble\Table\Columns\FormattedColumn;
use Botble\Table\Columns\IdColumn;
use Botble\Table\Columns\LinkableColumn;
use Botble\Table\Columns\StatusColumn;
use Illuminate\Database\Eloquent\Builder;

class OrderTable extends TableAbstract
{
    public function setup(): void
    {
        $this
            ->model(Order::class)
            ->addActions([
                EditAction::make()->route('events-booking.orders.edit'),
                DeleteAction::make()->route('events-booking.orders.destroy'),
            ])
            ->addColumns([
                IdColumn::make(),
                LinkableColumn::make('event_id')
                    ->urlUsing(function (LinkableColumn $column) {
                        $order = $column->getItem();

                        $order->loadMissing('event');

                        if (! $order->event) {
                            return null;
                        }

                        return route('events-booking.events.edit', $order->event_id);
                    })
                    ->label(trans('plugins/events-booking::events-booking.order.event'))
                    ->getValueUsing(function (FormattedColumn $column) {
                        $item = $column->getItem();
                        $item->loadMissing('event');

                        return $item->event_name;
                    }),
                LinkableColumn::make('account_id')
                    ->urlUsing(function (LinkableColumn $column) {
                        $order = $column->getItem();

                        if (! $order->account_id) {
                            return null;
                        }

                        return route('events-booking.accounts.edit', $order->account_id);
                    })
                    ->label(trans('plugins/events-booking::events-booking.order.account_name'))
                    ->getValueUsing(function (FormattedColumn $column) {
                        $item = $column->getItem();
                        $item->loadMissing('account');

                        if (! $account = $item->account) {
                            return $item->account_name ?: '-';
                        }

                        return $account->name;
                    }),
                FormattedColumn::make('price')
                    ->getValueUsing(function (FormattedColumn $column) {
                        $item = $column->getItem();

                        return format_price($item->price);
                    })
                    ->label(trans('plugins/events-booking::events-booking.order.price')),
                Column::make('quantity')
                    ->alignCenter()
                    ->label(trans('plugins/events-booking::events-booking.order.quantity')),
                FormattedColumn::make('total')
                    ->getValueUsing(function (FormattedColumn $column) {
                        $item = $column->getItem();

                        return format_price($item->total);
                    })
                    ->label(trans('plugins/events-booking::events-booking.order.total')),
                FormattedColumn::make('payment_id')
                    ->searchable(false)
                    ->label(trans('plugins/events-booking::events-booking.order.payment_status'))
                    ->getValueUsing(function (FormattedColumn $column) {
                        if (! is_plugin_active('payment')) {
                            return '&mdash;';
                        }

                        $item = $column->getItem();
                        $item->loadMissing('payment');

                        if (! $payment = $item->payment) {
                            return '-';
                        }

                        return $payment->status->toHtml();
                    }),
                StatusColumn::make(),
                CreatedAtColumn::make(),
            ])
            ->addBulkChanges([
                CreatedAtBulkChange::make(),
            ])
            ->addBulkActions([
                DeleteBulkAction::make()->permission('events-booking.orders.destroy'),
            ])
            ->queryUsing(function (Builder $query) {
                $query->select([
                    'id',
                    'event_id',
                    'event_name',
                    'account_id',
                    'payment_id',
                    'account_name',
                    'price',
                    'quantity',
                    'total',
                    'subtotal',
                    'status',
                    'created_at',
                ]);
            });
    }
}
