<?php

namespace ArchiElite\EventsBooking\Tables;

use ArchiElite\EventsBooking\Models\Order;
use Botble\Base\Facades\BaseHelper;
use Botble\Table\Abstracts\TableAbstract;
use Botble\Table\Columns\CreatedAtColumn;
use Botble\Table\Columns\FormattedColumn;
use Botble\Table\Columns\IdColumn;
use Botble\Table\Columns\LinkableColumn;
use Botble\Table\Columns\StatusColumn;
use Illuminate\Database\Eloquent\Builder;

class RecentBookingTable extends TableAbstract
{
    public function setup(): void
    {
        $columns = [
            IdColumn::make(),
            LinkableColumn::make('event_id')
                ->urlUsing(function (LinkableColumn $column) {
                    $order = $column->getItem();
                    $order->loadMissing('event');

                    if (! $order->event) {
                        return null;
                    }

                    return route('events-booking.events.edit', $order->event_id);
                })
                ->label(trans('plugins/events-booking::events-booking.order.event'))
                ->getValueUsing(function (FormattedColumn $column) {
                    $item = $column->getItem();
                    $item->loadMissing('event');

                    return $item->event_name;
                }),
            LinkableColumn::make('account_id')
                ->urlUsing(function (LinkableColumn $column) {
                    $order = $column->getItem();

                    if (! $order->account_id) {
                        return null;
                    }

                    return route('events-booking.accounts.edit', $order->account_id);
                })
                ->label(trans('plugins/events-booking::events-booking.order.account_name'))
                ->getValueUsing(function (FormattedColumn $column) {
                    $item = $column->getItem();
                    $item->loadMissing('account');

                    if (! $account = $item->account) {
                        return $item->account_name ?: '-';
                    }

                    return $account->name;
                }),
            FormattedColumn::make('total')
                ->getValueUsing(function (FormattedColumn $column) {
                    $item = $column->getItem();

                    return format_price($item->total);
                })
                ->title(trans('plugins/events-booking::booking.amount'))
                ->alignLeft(),
            CreatedAtColumn::make(),
        ];

        if (is_plugin_active('payment')) {
            $columns = array_merge($columns, [
                FormattedColumn::make('payment_id')
                    ->getValueUsing(function (FormattedColumn $column) {
                        $item = $column->getItem();

                        return BaseHelper::clean($item->payment->payment_channel->label() ?: '&mdash;');
                    })
                    ->name('payment_id')
                    ->title(trans('plugins/events-booking::booking.payment_method'))
                    ->alignLeft()
                    ->orderable(false)
                    ->searchable(false),
                FormattedColumn::make('payment_status')
                    ->getValueUsing(function (FormattedColumn $column) {
                        $item = $column->getItem();

                        return $item->payment->status->label() ? BaseHelper::clean(
                            $item->payment->status->toHtml()
                        ) : '&mdash;';
                    })
                    ->name('payment_status')
                    ->title(trans('plugins/events-booking::booking.payment_status_label'))
                    ->orderable(false)
                    ->searchable(false),
            ]);
        }

        $this
            ->model(Order::class)
            ->addColumns([...$columns, StatusColumn::make()])
            ->queryUsing(function (Builder $query) {
                $query = $query
                    ->select([
                        'id',
                        'event_name',
                        'event_id',
                        'account_id',
                        'account_name',
                        'total',
                        'created_at',
                        'status',
                        'payment_id',
                    ])
                    ->with(['event']);

                if (is_plugin_active('payment')) {
                    $query->with('payment');
                }

                return $query;
            });

        $this->type = self::TABLE_TYPE_SIMPLE;
        $this->defaultSortColumn = 0;
        $this->view = $this->simpleTableView();
    }
}
